#!/usr/bin/env python

from __future__ import print_function

import re;
import sys;
from math import floor;
from os import system;
import os;

from argparse import ArgumentParser, RawDescriptionHelpFormatter


def readFromDisk(device, start = 0, length = 0x10000):
    devSource = os.open(device, os.O_RDONLY)
    os.lseek(devSource, start, os.SEEK_SET)
    return os.read(devSource, length)

usage="""
"""

parser = ArgumentParser(description=usage,formatter_class=RawDescriptionHelpFormatter)
parser.add_argument('-v', dest='verbose', help='Turn on verbosity', action='count')
parser.add_argument('-L', dest='logicalSectorSize', default=512, help='logical sector size', type=int)
parser.add_argument('-s', dest='hardwareSectorSizeInLogicalSectors', default=8, help='hardware sector size in logical sectors', type=float)
parser.add_argument('-S', dest='sector', help='Sector', type=int, default=None)
parser.add_argument('-w', dest='wipe', help='wipe (set zero) sector if unreadable', action='store_true')
parser.add_argument('-m', dest='mirrorFrom', help='mirror data from disk or partition')
parser.add_argument('-f', dest='assumeFaulty', help='proceed on bad sectors even if readable!', action='store_true')
parser.add_argument('--startCheck', dest='startCheck', help='When mirroring check starting from this', default=(2048 * 512), type=int)
parser.add_argument('--dontTestFaulty', dest='testFaulty', help="Don't test sector faultyness.", action='store_false')
parser.add_argument("disks", nargs='+', help="The disks / partitions to be translated.")

options = parser.parse_args()

def verbose(text):
    if options.verbose:
        print(text, file=sys.stderr)

def warn(text):
    print("Warning: " + text, file=sys.stderr)

def error(text):
    print("Error: " + text, file=sys.stderr)


diskRe=re.compile('^(/dev/)?([hs]d[a-z]+)([0-9]*)$')

def isDiscOrParition(name):
    return bool(diskRe.match(name))

def parseDisc(name):
    matching = diskRe.match(name)
    if(matching):
        return (matching.group(2), matching.group(3))
    else :
        raise Exception("No disk or partition:" + name)


def isDisc(name):
    return isDiscOrParition(name) and parseDisc(name)[1] is None;

def file_get_contents(filename):
    with file(filename) as f:
        s = f.read()
    return s

def getSysFolder(name):
    disk, partNum = parseDisc(name)
    return '/sys/block/' + (name + '/' +  name + str(partNum) if partNum else name)  + '/'

def getSysFolder(name):
    disk, partNum = parseDisc(name) 
    return '/sys/block/' + (disk + '/' +  disk + str(partNum) if partNum else disk)  + '/'

    
def getAllPartitions():
    partitions = []
    with open('/proc/partitions') as f:
        data = f.readlines()
        i = 0
        for l in data:
            i+=1
            if i == 1 : continue
            
            parts = l.split();
            if (len(parts) == 4):
                block, name = parts[2:]
                if isDiscOrParition(name):
                    disk, partNum = parseDisc(name)
                    if partNum:
                        partitions.append(name); 
    #                 print "l=", l
#                     print "block=" + block, "name=" + name, "disk=" + disk, "partnum=" + partNum
    return partitions

class Partition:
    def __init__(self, name):
        self.name = name
        folder = getSysFolder(name)
        self.start = int(file_get_contents(folder + 'start').strip())
        self.size = int(file_get_contents(folder + 'size').strip())
        
        verbose("new partition " + str(self))

    def __str__(self):
        return self.name + "(start=%d, size=%d)" % (self.start, self.size)

    def getDiscSector(self, sectorInPartition):
        assert (sectorInPartition < self.size);
        return sectorInPartition + self.start;
    
    def getPartitionSector(self, sectorInDisc):
        s = sectorInDisc - self.start
        assert (s >= 0);
        assert (s < self.size);
        return s;

    def getStart(self):
        return self.start
    def getEnd(self):
        return self.start + self.size

class Disc:
    def __init__(self, name):
        self.name = name
        folder = getSysFolder(name)
        self.size = int(file_get_contents(folder + 'size').strip())
        self.partitions = dict()
        
    def addPartition(self, name, partNumm):
        self.partitions[partNum]=Partition(name)
        
    def getPartitionAndSector(self, sectorInDisc):
        assert (sectorInDisc < self.size);
        ret = []
        for p in self.partitions.values():
            if p.getStart() <= sectorInDisc and p.getEnd() > sectorInDisc:
                ret.append((p.name, p.getPartitionSector(sectorInDisc)))
        return ret

    def getDiscSector(self, partNum, sectorInPartition):
        return self.partitions[partNum].getDiscSector(sectorInPartition);

disksDb=dict();

for name in getAllPartitions():
    if isDiscOrParition(name):
        disk, partNum = parseDisc(name)
        if not disksDb.has_key(disk):
            disksDb[disk] = Disc(disk)

        if partNum:
            disksDb[disk].addPartition(name, partNum)
        else:
            verbose("disk: name=" + name + " size=" + size)
    else:
        warn("No disk : " + name)


def logicalToHardware(logicalSector):
    return floor(logicalSect / options.hardwareSectorSizeInLogicalSectors)


if options.sector >= 0:
    disks = options.disks
    sector = options.sector
    for name in disks:
        disk, partNum = parseDisc(name)
        if disk in disksDb:
            d = disksDb[disk]; 
            if partNum:
                verbose("Sector %d in partition %s%s:" % (sector, disk, partNum))
                print(d.getDiscSector(partNum, sector))
            else:
                verbose("Sector %d in disk %s:" % (sector, disk))
                for d in d.getPartitionAndSector(sector):
                    print(":".join([ str(a) for a in d]))
        else:
            verbose("No partitions in disk %s." % disk)

        if options.wipe or options.mirrorFrom :
            if partNum:
                logicalSect = d.getDiscSector(partNum, sector)
            else:
                logicalSect = sector
                
            hardwareSect = logicalToHardware(logicalSect)
            hardwareSectSize = int(options.logicalSectorSize * options.hardwareSectorSizeInLogicalSectors)
            verbose('Going to analyze sector L:%d,H:%d  (size = %d).' % (logicalSect, hardwareSect, hardwareSectSize))
            
            if options.testFaulty:
                ret = system('dd status=none of=/dev/null if=/dev/%s count=1 bs=%d skip=%d' % (disk, hardwareSectSize, hardwareSect))
            else : 
                ret =0
            if(ret != 0 or options.assumeFaulty):
                if ret != 0:
                    warn("This sector seems indeed faulty");
                else :
                    warn("This sector seems fine. BUT assuming faulty because instructed to do so!");
                    
                if options.mirrorFrom :
                    startCheckFrom = options.startCheck;
                    verbose("Checking correspondence between %s and %s starting at %d:" % (options.mirrorFrom, name, startCheckFrom))
                    
                    sourceData = readFromDisk(options.mirrorFrom, startCheckFrom)
                    destData = readFromDisk(name, startCheckFrom)
                    if(sourceData != destData):
                        error("Data doesn't match")
                    else:
                        verbose("Data matches.")
                        if(sourceData.count("\x00") == len(sourceData)):
                            error('Found only zeros! Check not meaningul!')
                        else:
                            warn('Use the following to overwrite!')
                            sect = int(hardwareSect * options.hardwareSectorSizeInLogicalSectors)
                            if partNum :
                                found = d.getPartitionAndSector(sect)
                                assert(len(found) == 1)
                                part,sect = found[0]
                                #print (found)
                                assert('/dev/' + part == name)
                            print ('dd if=%s of=/dev/%s count=%d ibs=%d obs=%d seek=%d skip=%d' % (options.mirrorFrom, disk, options.hardwareSectorSizeInLogicalSectors, options.logicalSectorSize, hardwareSectSize, hardwareSect, sect))

                else:
                    warn('Use the following to overwrite!')
                    print ('dd if=/dev/zero of=/dev/%s count=1 bs=%d seek=%d' % (disk, hardwareSectSize, hardwareSect))
