#define _LARGEFILE64_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <string>
#include <vector>
#include <chrono>
#include <thread>
#include <iostream>
#include <cstdlib>

#define BLOCKSIZE 512

int count;
time_t start;
int timeout = 100;

void done(int)
{
	time_t end;

	time(&end);

	if (end < start + timeout) {
		printf(".");
		alarm(1);
		return;
	}

	if (count) {
	  printf(".\nResults: %d seeks/second, %.2f ms random access time\n",
		 count / timeout, 1000.0 * timeout / count);
	}
	exit(EXIT_SUCCESS);
}

void handle(std::string string, int error)
{
	if (error) {
		printf(".\n");
		perror(string.c_str());
		exit(EXIT_FAILURE);
	}
}

void usage(){
	printf("Usage: disk-signaler [-v] [-h] [-t <timeout>] <raw disk device>...\n");
	exit(EXIT_SUCCESS);
}

int main(int argc, char **argv)
{
	char buffer[BLOCKSIZE];
	bool verbose = false;

	setvbuf(stdout, NULL, _IONBF, 0);

	int c;
	while ((c = getopt (argc, argv, "vht:")) != -1){
		switch (c)
		{
			case 'v':
				verbose = true;
				break;
			case 't':
				timeout = std::atoi(optarg);
				break;
			case 'h':
			case '?':
			default:
				usage();
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1) {
		usage();
	}

	std::vector<int> fds;
	std::vector<std::int64_t> numblocks;

	for(int i = 0; i < argc; i++){
		const char * device = argv[i];
		const int fd = open(device, O_RDONLY);
		handle(std::string("open ") + device, fd < 0);
		std::int64_t numblock;
		int retval = ioctl(fd, BLKGETSIZE, &numblock);
		handle(std::string("ioctl ") + device, retval == -1);
		printf("Benchmarking %s [%luMB], wait %d seconds:\n",
			device, numblock / 2048, timeout);
		fds.push_back(fd);
		numblocks.push_back(numblock);
	}

	time(&start);
	srand(start);
	signal(SIGALRM, &done);
	alarm(1);

	unsigned i = 0;
	auto waitTime = std::chrono::microseconds(1000000 / fds.size());
	printf("Waiting between each of %lu disks for %lu us.\n", fds.size(), waitTime.count());
	for (;;) {
		const double randFrac = (double) random() / RAND_MAX;
		const off64_t offset = (off64_t) ((double) numblocks[i] * randFrac);
		if(verbose) {
			printf("%f->%li (%d->%s), ", randFrac, offset, i, argv[i]);
		}
		const auto fd = fds[i];
		int retval = lseek64(fd, BLOCKSIZE * offset, SEEK_SET);
		handle(std::string("lseek64") + argv[i], retval == (off64_t) -1);
		retval = read(fd, buffer, BLOCKSIZE);
		handle(std::string("read ") + argv[i], retval < 0);
		std::this_thread::sleep_for(waitTime);
		count++;
		i = (i + 1) % fds.size();
	}
}

